<?php
// cron/retention_cleanup.php

require __DIR__ . '/../core/db.php';
require __DIR__ . '/../core/storage/StorageFactory.php';

echo "Retention cleanup started...\n";

/*
 * Fetch all users with packages + retention rules
 */
$sql = "
SELECT 
    up.user_id,
    p.retention_versions,
    p.retention_days
FROM user_packages up
JOIN packages p ON p.id = up.package_id
WHERE up.user_id IS NOT NULL
";

$users = $pdo->query($sql)->fetchAll(PDO::FETCH_ASSOC);

foreach ($users as $u) {

    $userId = (int)$u['user_id'];

    // IMPORTANT: treat NULL or 0 as unlimited
    $maxVersions = !empty($u['retention_versions'])
        ? (int)$u['retention_versions']
        : 0;

    $maxDays = !empty($u['retention_days'])
        ? (int)$u['retention_days']
        : 0;

    // -------------------------------------------------
    // UNLIMITED RETENTION → SKIP USER
    // -------------------------------------------------
    if ($maxVersions <= 0 && $maxDays <= 0) {
        echo "User {$userId}: unlimited retention, skipped\n";
        continue;
    }

    /*
     * Get all files grouped by virtual_path
     */
    $files = $pdo->prepare("
        SELECT DISTINCT virtual_path
        FROM backup_files
        WHERE user_id = ?
    ");
    $files->execute([$userId]);

    foreach ($files as $f) {

        $virtual = $f['virtual_path'];

        /*
         * Fetch versions DESC (latest first)
         */
        $versions = $pdo->prepare("
            SELECT *
            FROM backup_files
            WHERE user_id = ?
              AND virtual_path = ?
            ORDER BY version_no DESC, created_at DESC
        ");
        $versions->execute([$userId, $virtual]);
        $rows = $versions->fetchAll(PDO::FETCH_ASSOC);

        // Always keep at least one version
        if (count($rows) <= 1) {
            continue;
        }

        $toDelete = [];

        // -------- RULE 1: KEEP LAST N VERSIONS --------
        if ($maxVersions > 0 && count($rows) > $maxVersions) {
            $toDelete = array_slice($rows, $maxVersions);
        }

        // -------- RULE 2: OLDER THAN X DAYS --------
        if ($maxDays > 0) {
            $cutoff = strtotime("-{$maxDays} days");
            foreach ($rows as $r) {
                if (strtotime($r['created_at']) < $cutoff) {
                    $toDelete[] = $r;
                }
            }
        }

        // Remove duplicates safely
        $toDelete = array_unique($toDelete, SORT_REGULAR);

        foreach ($toDelete as $del) {

            // Never delete latest version
            if ($del['version_no'] == $rows[0]['version_no']) {
                continue;
            }

            /*
             * Delete from storage
             */
            $srv = $pdo->prepare("SELECT * FROM backup_servers WHERE id=?");
            $srv->execute([$del['server_id']]);
            $server = $srv->fetch(PDO::FETCH_ASSOC);

            if ($server) {
                try {
                    $storage = getStorage($server);
                    $storage->delete($del['path']);
                } catch (Exception $e) {
                    // Skip delete failure, do NOT stop cron
                    continue;
                }
            }

            /*
             * Delete DB record
             */
            $pdo->prepare("
                DELETE FROM backup_files WHERE id = ?
            ")->execute([$del['id']]);

            /*
             * Update quota usage
             */
            $pdo->prepare("
                UPDATE user_packages
                SET used_bytes = GREATEST(used_bytes - ?, 0)
                WHERE user_id = ?
            ")->execute([$del['size'], $userId]);

            echo "Deleted {$del['path']} (user {$userId})\n";
        }
    }
}

echo "Retention cleanup completed.\n";
