<?php
// backup/upload.php
header("Content-Type: application/json");

require __DIR__ . "/../auth_check.php";
require __DIR__ . "/../core/storage/StorageFactory.php";
require __DIR__ . "/../core/services/NotificationService.php";
require __DIR__ . "/../core/security/DeviceIsolationService.php";

/* -------------------------------------------------
 * BASIC VALIDATION
 * ------------------------------------------------- */
if (!isset($_FILES['file'])) {
    http_response_code(400);
    exit(json_encode(["error" => "No file uploaded"]));
}

$userId   = (int)$auth['uid'];
$deviceId = $auth['device_id'] ?? null;
$file     = $_FILES['file'];

if ($file['error'] !== UPLOAD_ERR_OK) {
    http_response_code(400);
    exit(json_encode(["error" => "Upload error"]));
}

$tmpPath  = $file['tmp_name'];
$fileName = basename($file['name']);
$fileSize = filesize($tmpPath);

if ($fileSize <= 0) {
    http_response_code(400);
    exit(json_encode(["error" => "Empty file"]));
}

$uploadMode = in_array($_POST['upload_mode'] ?? 'sync', ['sync','manual'], true)
    ? $_POST['upload_mode']
    : 'sync';

$virtualPath = trim($_POST['virtual_path'] ?? $fileName, '/');
$virtualPath = $virtualPath ?: $fileName;

/* -------------------------------------------------
 * QUOTA + EXPIRY CHECK
 * ------------------------------------------------- */
$stmt = $pdo->prepare("
    SELECT 
        (p.quota_gb * 1024 * 1024 * 1024) AS quota_bytes,
        IFNULL(up.used_bytes,0) AS used_bytes,
        up.expires_on
    FROM user_packages up
    JOIN packages p ON p.id = up.package_id
    WHERE up.user_id = ?
");
$stmt->execute([$userId]);
$pkg = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$pkg) {
    http_response_code(403);
    exit(json_encode(["error" => "No active package"]));
}

if (!empty($pkg['expires_on']) && strtotime($pkg['expires_on']) < time()) {
    http_response_code(403);
    exit(json_encode(["error" => "Subscription expired"]));
}

if ($pkg['used_bytes'] + $fileSize > $pkg['quota_bytes']) {
    http_response_code(403);
    exit(json_encode(["error" => "Quota exceeded"]));
}

/* -------------------------------------------------
 * RANSOMWARE HEURISTIC (ENTROPY)
 * ------------------------------------------------- */
function entropy(string $file): float {
    $data = file_get_contents($file, false, null, 0, 65536);
    if (!$data) return 0;
    $freq = count_chars($data, 1);
    $len  = strlen($data);
    $h = 0;
    foreach ($freq as $c) {
        $p = $c / $len;
        $h -= $p * log($p, 2);
    }
    return $h;
}

$entropyVal   = entropy($tmpPath);
$isSuspicious = ($entropyVal > 7.8);

if ($isSuspicious && $deviceId) {

    // 🔒 Isolate only this device
    DeviceIsolationService::block(
        $pdo,
        (int)$deviceId,
        'High-entropy upload detected (possible ransomware)'
    );

    NotificationService::push($pdo, [
        'user_id'  => $userId,
        'title'    => 'Device Blocked',
        'message'  => 'One of your devices was blocked due to suspicious activity.',
        'severity' => 'critical'
    ]);

    NotificationService::notifyAdmins(
        $pdo,
        'Device Isolated',
        "Device {$deviceId} isolated for user {$userId} (high entropy upload)"
    );

    http_response_code(403);
    exit(json_encode([
        "error" => "This device has been blocked for security reasons"
    ]));
}

/* -------------------------------------------------
 * MASS CHANGE DETECTION (5 MIN WINDOW)
 * ------------------------------------------------- */
$stmt = $pdo->prepare("
    SELECT COUNT(*) 
    FROM backup_files
    WHERE user_id = ?
      AND created_at > (NOW() - INTERVAL 5 MINUTE)
");
$stmt->execute([$userId]);
$recent = (int)$stmt->fetchColumn();

if ($recent > 50 && $deviceId) {

    DeviceIsolationService::block(
        $pdo,
        (int)$deviceId,
        'Excessive file changes in short time window'
    );

    NotificationService::push($pdo, [
        'user_id'  => $userId,
        'title'    => 'Device Blocked',
        'message'  => 'A device was blocked due to unusual backup activity.',
        'severity' => 'critical'
    ]);

    NotificationService::notifyAdmins(
        $pdo,
        'Device Auto-Isolated',
        "Device {$deviceId} auto-blocked for user {$userId} (rate anomaly)"
    );

    http_response_code(403);
    exit(json_encode([
        "error" => "Device blocked due to abnormal activity"
    ]));
}

/* -------------------------------------------------
 * VERSIONING
 * ------------------------------------------------- */
$stmt = $pdo->prepare("
    SELECT IFNULL(MAX(version_no),0)
    FROM backup_files
    WHERE user_id = ? AND virtual_path = ?
");
$stmt->execute([$userId, $virtualPath]);
$versionNo = ((int)$stmt->fetchColumn()) + 1;

/* -------------------------------------------------
 * STORAGE SELECTION
 * ------------------------------------------------- */
$servers = $pdo->query("
    SELECT * FROM backup_servers
    ORDER BY FIELD(type,'sftp','ftp','http','s3')
")->fetchAll();

$storage = null;
$activeServer = null;

foreach ($servers as $s) {
    try {
        $storage = getStorage($s);
        $activeServer = $s;
        break;
    } catch (Exception $e) {}
}

if (!$storage) {
    http_response_code(500);
    exit(json_encode(["error" => "No storage backend available"]));
}

$remotePath = "/{$userId}/{$virtualPath}";

/* -------------------------------------------------
 * UPLOAD FILE
 * ------------------------------------------------- */
try {
    $storage->upload($tmpPath, $remotePath);
} catch (Exception $e) {
    http_response_code(500);
    exit(json_encode(["error" => "Storage upload failed"]));
}

/* -------------------------------------------------
 * DATABASE LOGGING
 * ------------------------------------------------- */
$pdo->prepare("
    INSERT INTO backup_files
        (user_id, server_id, path, virtual_path, upload_mode, size, version_no, created_at)
    VALUES
        (?, ?, ?, ?, ?, ?, ?, NOW())
")->execute([
    $userId,
    $activeServer['id'],
    $remotePath,
    $virtualPath,
    $uploadMode,
    $fileSize,
    $versionNo
]);

$pdo->prepare("
    UPDATE user_packages
    SET used_bytes = used_bytes + ?
    WHERE user_id = ?
")->execute([$fileSize, $userId]);

/* -------------------------------------------------
 * QUOTA WARNING
 * ------------------------------------------------- */
$usedAfter = $pkg['used_bytes'] + $fileSize;
$percent   = ($usedAfter / $pkg['quota_bytes']) * 100;

if ($percent >= 90) {
    NotificationService::push($pdo, [
        'user_id'  => $userId,
        'title'    => 'Storage Almost Full',
        'message'  => 'You have used more than 90% of your storage quota.',
        'severity' => 'warning'
    ]);
}

/* -------------------------------------------------
 * SUCCESS
 * ------------------------------------------------- */
echo json_encode([
    "status"  => "success",
    "file"    => $fileName,
    "version" => $versionNo
]);
