<?php
// api/auth_check.php
header("Content-Type: application/json");

require __DIR__ . "/core/db.php";
require __DIR__ . "/core/jwt.php";

/* -------------------------------------------------
 * 1. Get Authorization header
 * ------------------------------------------------- */
$headers = getallheaders();

if (empty($headers['Authorization'])) {
    http_response_code(401);
    exit(json_encode(["error" => "Missing Authorization header"]));
}

if (!preg_match('/Bearer\s(\S+)/', $headers['Authorization'], $matches)) {
    http_response_code(401);
    exit(json_encode(["error" => "Invalid Authorization format"]));
}

$token = $matches[1];

/* -------------------------------------------------
 * 2. Decode JWT
 * ------------------------------------------------- */
$auth = jwt_decode($token);

if (
    !$auth ||
    empty($auth['uid']) ||
    empty($auth['device_token'])
) {
    http_response_code(401);
    exit(json_encode(["error" => "Invalid or expired token"]));
}

$userId      = (int)$auth['uid'];
$deviceToken = $auth['device_token'];

/* -------------------------------------------------
 * 3. Validate user + subscription
 * ------------------------------------------------- */
$stmt = $pdo->prepare("
    SELECT 
        u.id,
        u.status,
        up.expires_on
    FROM users u
    LEFT JOIN user_packages up ON up.user_id = u.id
    WHERE u.id = ?
    LIMIT 1
");
$stmt->execute([$userId]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$user) {
    http_response_code(403);
    exit(json_encode(["error" => "User not found"]));
}

/* ---------- Account status ---------- */
if ((int)$user['status'] !== 1) {
    http_response_code(403);
    exit(json_encode(["error" => "Account disabled"]));
}

/* ---------- Subscription expiry ---------- */
if (!empty($user['expires_on']) && strtotime($user['expires_on']) < time()) {
    http_response_code(403);
    exit(json_encode(["error" => "Subscription expired"]));
}
/*-------------------------------------------------
 * 3.5. DEVICE ISSOLATION - CHECK IF DEVICE IS BLOCKED
 * ------------------------------------------------- */
$deviceId = $auth['device_id'] ?? null;

if ($deviceId) {
    $stmt = $pdo->prepare("
        SELECT blocked FROM user_devices
        WHERE id = ? AND user_id = ?
    ");
    $stmt->execute([$deviceId, $userId]);
    $device = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($device && (int)$device['blocked'] === 1) {
        http_response_code(403);
        exit(json_encode([
            "error" => "This device has been blocked for security reasons"
        ]));
    }
}


/* -------------------------------------------------
 * 4. DEVICE ISOLATION CHECK (CRITICAL)
 * ------------------------------------------------- */
$stmt = $pdo->prepare("
    SELECT 
        id,
        blocked
    FROM user_devices
    WHERE user_id = ?
      AND auth_token = ?
    LIMIT 1
");
$stmt->execute([$userId, $deviceToken]);
$device = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$device) {
    http_response_code(403);
    exit(json_encode(["error" => "Unknown device"]));
}

/* ---------- Device blocked ---------- */
if ((int)$device['blocked'] === 1) {
    http_response_code(403);
    exit(json_encode([
        "error" => "Device quarantined due to security risk"
    ]));
}

/* -------------------------------------------------
 * 5. Update device heartbeat (last_seen)
 * ------------------------------------------------- */
$pdo->prepare("
    UPDATE user_devices
    SET last_seen = NOW()
    WHERE id = ?
")->execute([$device['id']]);

/* -------------------------------------------------
 * AUTH PASSED
 * -------------------------------------------------
 * Available variables:
 *
 * $auth        → JWT payload
 * $userId      → authenticated user ID
 * $deviceToken → device auth token
 * $device['id']→ device ID
 *
 * ------------------------------------------------- */
